#!/usr/bin/env python
#
# SismoVi, version 1.05, September 2004.
#      
# Copyright (C) 2004, Francesco Apostolico, Geza Seriani
# This program is free software you can redistribute it and/or 
#     modify it under the terms of the GNU General Public License 
#     as published by the Free Software Foundation either version 2
#     of the License, or any later version, with the following conditions
#     attached in addition to any and all conditions of the GNU
#     General Public License:
#     When reporting or displaying any results or animations created
#     using this code or modification of this code, make the appropriate
#     citation referencing SismoVi by name and including the version
#     number.
#
# This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY without even the implied warranty 
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#     See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
#     along with this program if not, write to the Free Software
#     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  
#     02111-1307  USA
#
# Contacting the authors:
#
# Francesco Apostolico, Geza Seriani
# Istituto Nazionale di Oceanografia e di Geofisica Sperimentale - OGS
# Borgo Grotta Gigante, 42/C, Sgonico
# 34010, Trieste, Italy
#
# fapostolico@inogs.it, gseriani@inogs.it
# 
# #######################################/
# SismoVi is a simple tool for the visualization of seismograms.
#      It is platform independent and it is based on 
#      Python (http://www.python.org/), Numeric (http://www.pfdubois.com/numpy/) 
#      and  WxPython 2.5.1 (http://wxpython.org/) that must be installed in your 
#      platform before running it. 
#      An interactive help is available when running SismoVi.
#

import wx, wx.grid as grid

class SismoDialog(wx.Dialog):
    def __init__(self, *args, **kwds):
        # begin wxGlade: MyDialog.__init__
        #kwds["style"] = wx.DEFAULT_DIALOG_STYLE
        wx.Dialog.__init__(self, *args, **kwds)
        self.label_1 = wx.StaticText(self, -1, "General data file informations")
        self.static_line_1 = wx.StaticLine(self, -1)
        self.label_25 = wx.StaticText(self, -1, "Title:")
        self.text_ctrl_11 = wx.TextCtrl(self, -1, "")
        self.panel_13 = wx.Panel(self, -1)
        self.panel_14 = wx.Panel(self, -1)
        self.label_26 = wx.StaticText(self, -1, "X label:            ")
        self.text_ctrl_12 = wx.TextCtrl(self, -1, "")
        self.label_27 = wx.StaticText(self, -1, "Y label:            ")
        self.text_ctrl_13 = wx.TextCtrl(self, -1, "")
        self.static_line_7 = wx.StaticLine(self, -1)
        self.label_2 = wx.StaticText(self, -1, "Traces:")
        self.spin_ctrl_1 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_3 = wx.StaticText(self, -1, "Trace step:")
        self.text_ctrl_1 = wx.TextCtrl(self, -1, "")
        self.label_4 = wx.StaticText(self, -1, "Samples:")
        self.spin_ctrl_2 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_5 = wx.StaticText(self, -1, "Time step:")
        self.text_ctrl_2 = wx.TextCtrl(self, -1, "")
        self.label_6 = wx.StaticText(self, -1, "Peak value:")
        self.text_ctrl_3 = wx.TextCtrl(self, -1, "")
        self.panel_1 = wx.Panel(self, -1)
        self.panel_2 = wx.Panel(self, -1)
        self.label_7 = wx.StaticText(self, -1, "Average value:")
        self.text_ctrl_4 = wx.TextCtrl(self, -1, "")
        self.panel_3 = wx.Panel(self, -1)
        self.panel_4 = wx.Panel(self, -1)
        self.static_line_2 = wx.StaticLine(self, -1)
        self.label_8 = wx.StaticText(self, -1, "Trace set visualization parameters")
        self.static_line_3 = wx.StaticLine(self, -1)
        self.label_9 = wx.StaticText(self, -1, "First trace:")
        self.spin_ctrl_3 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_10 = wx.StaticText(self, -1, "Last trace:")
        self.spin_ctrl_4 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_11 = wx.StaticText(self, -1, "Decimation factor:")
        self.spin_ctrl_5 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.panel_5 = wx.Panel(self, -1)
        self.panel_6 = wx.Panel(self, -1)
        self.label_12 = wx.StaticText(self, -1, "Clip (%) value:")
        self.spin_ctrl_6 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_13 = wx.StaticText(self, -1, "Overlap (%) value:")
        self.spin_ctrl_7 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.static_line_4 = wx.StaticLine(self, -1)
        self.label_14 = wx.StaticText(self, -1, "First sample:")
        self.spin_ctrl_8 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_15 = wx.StaticText(self, -1, "Last sample:         ")
        self.spin_ctrl_9 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.label_16 = wx.StaticText(self, -1, "Decimation factor:")
        self.spin_ctrl_10 = wx.SpinCtrl(self, -1, "", min=0, max=100)
        self.panel_7 = wx.Panel(self, -1)
        self.panel_8 = wx.Panel(self, -1)
        self.static_line_5 = wx.StaticLine(self, -1)
        self.label_17 = wx.StaticText(self, -1, "Window parameters")
        self.static_line_6 = wx.StaticLine(self, -1)
        self.label_18 = wx.StaticText(self, -1, "Space init.:")
        self.text_ctrl_5 = wx.TextCtrl(self, -1, "")
        self.label_19 = wx.StaticText(self, -1, "Space length:     ")
        self.text_ctrl_6 = wx.TextCtrl(self, -1, "")
        self.label_20 = wx.StaticText(self, -1, "Tick scale:     ")
        self.text_ctrl_7 = wx.TextCtrl(self, -1, "")
        self.panel_9 = wx.Panel(self, -1)
        self.panel_10 = wx.Panel(self, -1)
        self.label_21 = wx.StaticText(self, -1, "Time init.:")
        self.text_ctrl_8 = wx.TextCtrl(self, -1, "")
        self.label_22 = wx.StaticText(self, -1, "Time length:")
        self.text_ctrl_9 = wx.TextCtrl(self, -1, "")
        self.label_23 = wx.StaticText(self, -1, "Tick scale:")
        self.text_ctrl_10 = wx.TextCtrl(self, -1, "")
        self.panel_11 = wx.Panel(self, -1)
        self.panel_12 = wx.Panel(self, -1)
        self.static_line_9 = wx.StaticLine(self, -1)
        self.label_24 = wx.StaticText(self, -1, "Plot Style")
        self.radio_box_1 = wx.RadioBox(self, -1, "", choices=["Wiggle", "Variable area", "Wiggle+Variable area"], majorDimension=0, style=wx.RA_SPECIFY_COLS)
        self.button_1 = wx.Button(self, -1, "Draw")
        self.button_2 = wx.Button(self, -1, "Reset")
        self.button_3 = wx.Button(self, -1, "Hidden")
        self.static_line_10 = wx.StaticLine(self, -1)
        self.grid_1 = grid.Grid(self, -1)
        self.grid_1.CreateGrid(10,10)

        self.__set_properties()
        self.__do_layout()

        wx.EVT_CLOSE(self, self.OnCloseWindow)
        wx.EVT_BUTTON(self, self.button_1.GetId(), self.OnSubmit)
        wx.EVT_BUTTON(self, self.button_2.GetId(), self.OnReset)
        wx.EVT_BUTTON(self, self.button_3.GetId(), self.OnCloseWindow)
        grid.EVT_GRID_CELL_CHANGE(self, self.OnCellChange)
        self.frame=args[0]

    def __set_properties(self):
        self.SetTitle("Input parameters")
        self.radio_box_1.SetSelection(0)
        self.button_1.SetFocus()
        self.button_1.SetDefault()
        self.button_2.SetFocus()

    def __do_layout(self):
        grid_sizer_1 = wx.FlexGridSizer(24, 1, 0, 0)
        grid_sizer_7 = wx.GridSizer(1, 3, 0, 0)
        grid_sizer_5 = wx.FlexGridSizer(4, 4, 0, 0)
        grid_sizer_4 = wx.FlexGridSizer(2, 4, 0, 0)
        grid_sizer_3 = wx.FlexGridSizer(3, 4, 0, 0)
        grid_sizer_2 = wx.FlexGridSizer(4, 4, 0, 0)
        grid_sizer_6 = wx.FlexGridSizer(2, 4, 0, 0)
        grid_sizer_1.Add(self.label_1, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(self.static_line_1, 0, wx.EXPAND, 0)
        grid_sizer_6.Add(self.label_25, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_6.Add(self.text_ctrl_11, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_6.Add(self.panel_13, 1, wx.EXPAND, 0)
        grid_sizer_6.Add(self.panel_14, 1, wx.EXPAND, 0)
        grid_sizer_6.Add(self.label_26, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_6.Add(self.text_ctrl_12, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_6.Add(self.label_27, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_6.Add(self.text_ctrl_13, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(grid_sizer_6, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(self.static_line_7, 0, wx.EXPAND, 0)
        grid_sizer_2.Add(self.label_2, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.spin_ctrl_1, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.label_3, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.text_ctrl_1, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.label_4, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.spin_ctrl_2, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.label_5, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.text_ctrl_2, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.label_6, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.text_ctrl_3, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.panel_1, 1, wx.EXPAND, 0)
        grid_sizer_2.Add(self.panel_2, 1, wx.EXPAND, 0)
        grid_sizer_2.Add(self.label_7, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.text_ctrl_4, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_2.Add(self.panel_3, 1, wx.EXPAND, 0)
        grid_sizer_2.Add(self.panel_4, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(grid_sizer_2, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(self.static_line_2, 0, wx.EXPAND, 0)
        grid_sizer_1.Add(self.label_8, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(self.static_line_3, 0, wx.EXPAND, 0)
        grid_sizer_3.Add(self.label_9, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.spin_ctrl_3, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.label_10, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.spin_ctrl_4, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.label_11, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.spin_ctrl_5, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.panel_5, 1, wx.EXPAND, 0)
        grid_sizer_3.Add(self.panel_6, 1, wx.EXPAND, 0)
        grid_sizer_3.Add(self.label_12, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.spin_ctrl_6, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.label_13, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_3.Add(self.spin_ctrl_7, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(grid_sizer_3, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(self.static_line_4, 0, wx.EXPAND, 0)
        grid_sizer_4.Add(self.label_14, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_4.Add(self.spin_ctrl_8, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_4.Add(self.label_15, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_4.Add(self.spin_ctrl_9, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_4.Add(self.label_16, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_4.Add(self.spin_ctrl_10, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_4.Add(self.panel_7, 1, wx.EXPAND, 0)
        grid_sizer_4.Add(self.panel_8, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(grid_sizer_4, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(self.static_line_5, 0, wx.EXPAND, 0)
        grid_sizer_1.Add(self.label_17, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(self.static_line_6, 0, wx.EXPAND, 0)
        grid_sizer_5.Add(self.label_18, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.text_ctrl_5, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.label_19, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.text_ctrl_6, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.label_20, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.text_ctrl_7, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.panel_9, 1, wx.EXPAND, 0)
        grid_sizer_5.Add(self.panel_10, 1, wx.EXPAND, 0)
        grid_sizer_5.Add(self.label_21, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.text_ctrl_8, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.label_22, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.text_ctrl_9, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.label_23, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.text_ctrl_10, 0, wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_5.Add(self.panel_11, 1, wx.EXPAND, 0)
        grid_sizer_5.Add(self.panel_12, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(grid_sizer_5, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(self.static_line_9, 0, wx.EXPAND, 0)
        grid_sizer_1.Add(self.label_24, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(self.radio_box_1, 0, wx.EXPAND, 0)
        grid_sizer_7.Add(self.button_1, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_7.Add(self.button_2, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_7.Add(self.button_3, 0, wx.ALIGN_CENTER_HORIZONTAL|wx.ALIGN_CENTER_VERTICAL, 0)
        grid_sizer_1.Add(grid_sizer_7, 1, wx.EXPAND, 0)
        grid_sizer_1.Add(self.static_line_10, 0, wx.EXPAND, 0)
        (h,v)=grid_sizer_1.GetSize()
        self.grid_1.SetSize((h,200))
        grid_sizer_1.Add(self.grid_1, 0, wx.EXPAND, 0)
        self.SetAutoLayout(1)
        self.SetSizer(grid_sizer_1)
        grid_sizer_1.Fit(self)
        grid_sizer_1.SetSizeHints(self)
        self.Layout()
        self.Centre()

    def OnCloseWindow(self, event):
        self.Show(False)

    def OnSubmit(self, event):
        try:
            val=float(self.text_ctrl_1.GetValue())
            valtracestep=[]
            for i in range(self.spin_ctrl_1.GetValue()-1):valtracestep.append(val)
        except:
            valtracestep=eval(self.text_ctrl_1.GetValue())
        try:
            val=float(self.text_ctrl_2.GetValue())
            valtimestep=[]
            for i in range(self.spin_ctrl_2.GetValue()-1):valtimestep.append(val)
        except:
            valtimestep=eval(self.text_ctrl_2.GetValue())
        par={'traces':int(self.spin_ctrl_1.GetValue()),
             'samples':int(self.spin_ctrl_2.GetValue()),
             'tracestep':valtracestep,
             'timestep':valtimestep,
             'peak':float(self.text_ctrl_3.GetValue()),
             'average':float(self.text_ctrl_4.GetValue()),
             'firsttrace':int(self.spin_ctrl_3.GetValue()),
             'lasttrace':int(self.spin_ctrl_4.GetValue()),
             'tracedec':int(self.spin_ctrl_5.GetValue()),
             'clipping':int(self.spin_ctrl_6.GetValue()),
             'overlap':int(self.spin_ctrl_7.GetValue()),
             'firstsample':int(self.spin_ctrl_8.GetValue()),
             'lastsample':int(self.spin_ctrl_9.GetValue()),
             'sampledec':int(self.spin_ctrl_10.GetValue()),
             'spaceinit':float(self.text_ctrl_5.GetValue()),
             'spacelength':float(self.text_ctrl_6.GetValue()),
             'spacerescale':float(self.text_ctrl_7.GetValue()),
             'timeinit':float(self.text_ctrl_8.GetValue()),
             'timelength':float(self.text_ctrl_9.GetValue()),
             'timerescale':float(self.text_ctrl_10.GetValue()),
             'format':self.radio_box_1.GetSelection(),
             'plottitle':self.text_ctrl_11.GetValue(),
             'xlabel':self.text_ctrl_12.GetValue(),
             'ylabel':self.text_ctrl_13.GetValue(),
             'data':self.tx3
        }
        self.frame.client.Draw(TraceSet(par),
                         (par['timeinit']*par['timerescale'],(par['timeinit']+par['timelength'])*par['timerescale']),
                         (par['spaceinit']*par['spacerescale'],(par['spaceinit']+par['spacelength'])*par['spacerescale']))

    def OnReset(self, event):
        self.ResetValues()

    def SetValues(self, tx2):
        try:
            for j in tx2.get('data'):
                maxlen=len(j)
            tx3a=Numeric.zeros(maxlen*len(tx2.get('data')),'float')
            tx3a.shape=(len(tx2.get('data')),maxlen)
            for j in range(len(tx2.get('data'))):
                tx3a[j,0:len(tx2.get('data')[j])]=tx2.get('data')[j][0:len(tx2.get('data')[j])]

            (traces,samples)=Numeric.array(tx3a).shape
            tx2['traces']=traces
            tx2['samples']=samples
        except:
            tx3a=tx2.get('data')
            
        self.tx3=Numeric.array(tx3a).flat
        del tx2['data']
        self.tx2=tx2
        self.ResetValues()

    def ResetValues(self):
        txabs=Numeric.fabs(self.tx3)
        txmax=max(txabs)
        txave=Numeric.average(txabs)
        try:
            self.spin_ctrl_1.SetRange(0,int(self.tx2['traces']))
            self.spin_ctrl_1.SetValue(int(self.tx2['traces']))
        except:
            self.spin_ctrl_1.SetRange(0,9999999)
            self.spin_ctrl_1.SetValue(0)
        try:
            self.spin_ctrl_2.SetRange(0,int(self.tx2['samples']))
            self.spin_ctrl_2.SetValue(int(self.tx2['samples']))
        except:
            self.spin_ctrl_2.SetRange(0,9999999)
            self.spin_ctrl_2.SetValue(0)
        try:
            self.text_ctrl_1.SetValue(str(eval(str(self.tx2['tracestep']))))
        except:
            self.text_ctrl_1.SetValue('')
        try:
            self.text_ctrl_2.SetValue(str(eval(str(self.tx2['timestep']))))
        except:
            self.text_ctrl_2.SetValue('')
        try:
            self.text_ctrl_3.SetValue(str(float(self.tx2['peakvalue'])))
        except:
            self.text_ctrl_3.SetValue(str(txmax))
        try:
            self.text_ctrl_4.SetValue(str(float(self.tx2['averagevalue'])))
        except:
            self.text_ctrl_4.SetValue(str(txave))
        try:
            self.spin_ctrl_3.SetRange(0,int(self.tx2['traces'])-1)
        except:
            self.spin_ctrl_3.SetRange(0,9999999)
        self.spin_ctrl_3.SetValue(0)
        try:
            self.spin_ctrl_4.SetRange(0,int(self.tx2['traces'])-1)
            self.spin_ctrl_4.SetValue(int(self.tx2['traces'])-1)
        except:
            self.spin_ctrl_4.SetRange(0,9999999)
            self.spin_ctrl_4.SetValue(0)
        self.spin_ctrl_5.SetRange(0,int(self.tx2['traces']))
        self.spin_ctrl_5.SetValue(1)
        self.spin_ctrl_6.SetRange(0,99)
        try:
            self.spin_ctrl_6.SetValue(int(self.tx2['clip']))
        except:
            self.spin_ctrl_6.SetValue(0)
        self.spin_ctrl_7.SetRange(0,99)
        try:
            self.spin_ctrl_7.SetValue(int(self.tx2['overlap']))
        except:
            self.spin_ctrl_7.SetValue(0)
        try:
            self.spin_ctrl_8.SetRange(0,int(self.tx2['samples'])-1)
        except:
            self.spin_ctrl_8.SetRange(0,9999999)
        self.spin_ctrl_8.SetValue(0)
        try:
            self.spin_ctrl_9.SetRange(0,int(self.tx2['samples'])-1)
            self.spin_ctrl_9.SetValue(int(self.tx2['samples'])-1)
        except:
            self.spin_ctrl_9.SetRange(0,9999999)
            self.spin_ctrl_9.SetValue(0)
        self.spin_ctrl_10.SetRange(0,int(self.tx2['samples']))
        self.spin_ctrl_10.SetValue(1)
        try:
             self.text_ctrl_5.SetValue(str(float(-self.tx2['tracestep'])))
        except:
            self.text_ctrl_5.SetValue('')
        try:
            self.text_ctrl_6.SetValue(str(float((self.tx2['traces']+1)*self.tx2['tracestep'])))
        except:
            self.text_ctrl_6.SetValue('')
        self.text_ctrl_7.SetValue(str(1))
        self.text_ctrl_8.SetValue(str(0))
        try:
            self.text_ctrl_9.SetValue(str(float((self.tx2['samples']-1)*self.tx2['timestep'])))
        except:
            self.text_ctrl_9.SetValue('')
        self.text_ctrl_10.SetValue(str(1))
        try:
            self.text_ctrl_11.SetValue(str(self.tx2['title']))
        except:
            self.text_ctrl_11.SetValue('')
        try:
            self.text_ctrl_12.SetValue(str(self.tx2['xlabel']))
        except:
            self.text_ctrl_12.SetValue('')
        try:
            self.text_ctrl_13.SetValue(str(self.tx2['ylabel']))
        except:
            self.text_ctrl_13.SetValue('')
        self.grid_1.DeleteRows(0,self.grid_1.GetNumberRows())
        self.grid_1.DeleteCols(0,self.grid_1.GetNumberCols())
        self.grid_1.AppendRows(int(self.tx2['traces']))
        self.grid_1.AppendCols(int(self.tx2['samples']))
        for i in range(self.grid_1.GetNumberCols()):
            self.grid_1.SetColLabelValue(i,str(i+1))
        for j in range(int(self.tx2['traces'])):
            for i in range(int(self.tx2['samples'])):
                self.grid_1.SetCellValue(j,i,str(self.tx3[j*int(self.tx2['samples'])+i]))

    def OnCellChange(self, event):
        self.tx3[event.GetRow()*self.spin_ctrl_2.GetValue()+event.GetCol()]=float(self.grid_1.GetCellValue(event.GetRow(),event.GetCol()))


# end of class MyDialog

try: from wx.lib.plot import *
except: from wxPyPlot import *
import Numeric, os, string, sys, wx.html
import wx.html

class PolyPolygon(PolyPoints):
    """Class to define line type and style
        - All methods except __init__ are private.
    """

    _attributes = {'colour': 'black',
                   'width': 1,
                   'style': wx.SOLID,
                   'fillcolour': 'black',
                   'fillstyle': wx.SOLID,
                   'legend': ''}

    def __init__(self, points, **attr):
        """Creates PolyLine object
            points - sequence (array, tuple or list) of (x,y) points making up line
            **attr - key word attributes
                Defaults:
                    'colour'= 'black',          - wx.Pen Colour any wxNamedColour
                    'width'= 1,                 - Pen width
                    'style'= wx.SOLID,           - wx.Pen style
                    'legend'= ''                - Line Legend to display
        """
        PolyPoints.__init__(self, points, attr)

    def draw(self, dc, printerScale, coord= None):
        colour = self.attributes['colour']
        width = self.attributes['width'] * printerScale
        style= self.attributes['style']
        fillcolour= self.attributes['fillcolour']
        fillstyle= self.attributes['fillstyle']
        dc.SetPen(wx.Pen(wx.NamedColour(colour), int(width), style))
        if coord == None:
            dc.SetBrush(wx.Brush(wx.NamedColour(fillcolour),fillstyle))
            dc.DrawPolygon(self.scaled)
        else:
            dc.DrawPolygon(coord) #draw legend line

    def getSymExtent(self, printerScale):
        """Width and Height of Marker"""
        h= self.attributes['width'] * printerScale
        w= 5 * h
        return (w,h)

def TraceSet(par):
    samples=par['samples']
    traces=par['traces']
    timestep=par['timestep']
    tracestep=par['tracestep']
    peak=par['peak']
    ave=par['average']
    clipping=par['clipping']
    overlap=par['overlap']
    firstsample=par['firstsample']
    lastsample=par['lastsample']
    sampledec=par['sampledec']
    firsttrace=par['firsttrace']
    lasttrace=par['lasttrace']
    tracedec=par['tracedec']
    fmt=par['format']
    ptitle=par['plottitle']
    xlabel=par['xlabel']
    ylabel=par['ylabel']
    tx1=par['data']
    mrescale=Numeric.array([[par['timerescale'],0],[0,par['spacerescale']]])
    lines=[]
    mint=min(tracestep)
    tx1a=Numeric.clip(tx1,-peak*(100-clipping)*0.01,peak*(100-clipping)*0.01)/peak/2*mint/(100-clipping)*tracedec/(100-overlap)*10000
    #tnum=Numeric.arange(samples)*timestep
    tnum=Numeric.array([0])
    for k in timestep:
        tnum=Numeric.concatenate((tnum,[k+tnum[-1]]))
    tnum.shape=(samples,1)
    snum=Numeric.array([0])
    for k in tracestep:
        snum=Numeric.concatenate((snum,[k+snum[-1]]))
    for j in range(firsttrace,lasttrace+1,tracedec):
        tx2=tx1a[j*samples:(j+1)*samples]
        tx2.shape=(samples,1)
        tx3=Numeric.concatenate((tnum,tx2),1)
        #yzero=j*tracestep
        yzero=snum[j]
        for i in range(firstsample,(lastsample+1),sampledec):
            if i==firstsample: # apertura traccia
                ln1=[[0,yzero]]
            if (ln1[-1][1]-yzero)*tx3[i,1]>=0 or fmt==0: # stesso segno
                ln1.append([tx3[i,0],tx3[i,1]+yzero])
            else: # aggiunge attraversamento
                y=[(tx3[i,0]-ln1[-1][0])*(ln1[-1][1]-yzero)/(ln1[-1][1]-tx3[i,1]-yzero)+ln1[-1][0],yzero]
                ln1.append(y)
                ln2=Numeric.matrixmultiply(Numeric.array(ln1),mrescale)
                if tx3[i,1]<0:
                    lines.append(PolyPolygon(ln2, legend= 'Red Line', colour='black'))
                else:
                    if fmt!=1: lines.append(PolyLine(ln2, legend= 'Red Line', colour='black'))
                ln1=[y]
                ln1.append([tx3[i,0],tx3[i,1]+yzero])
            if i==lastsample: # aggiunge punto chiusura traccia
                ln1.append([tx3[i,0],yzero])
                ln2=Numeric.matrixmultiply(Numeric.array(ln1),mrescale)
                if Numeric.sum(ln2[:,1]-yzero)>0 and fmt!=0:
                    lines.append(PolyPolygon(ln2, legend= 'Red Line', colour='black'))
                else:
                    if fmt!=1: lines.append(PolyLine(ln2, legend= 'Red Line', colour='black'))
    return PlotGraphics(lines,ptitle, xlabel, ylabel)

class PlotFrame(wx.Frame):
    def __init__(self, parent, id, title):
        wx.Frame.__init__(self, parent, id, title,
                          wx.DefaultPosition, (600, 400))

        # Now Create the menu bar and items
        self.mainmenu = wx.MenuBar()

        menu = wx.Menu()
        menu.Append(240, 'Open File...', 'Open file for plotting')
        wx.EVT_MENU(self, 240, self.OnFileOpenFile)

        menu.Append(200, 'Page Setup...', 'Setup the printer page')
        wx.EVT_MENU(self, 200, self.OnFilePageSetup)

        menu.Append(201, 'Print Preview...', 'Show the current plot on page')
        wx.EVT_MENU(self, 201, self.OnFilePrintPreview)

        menu.Append(202, 'Print...', 'Print the current plot')
        wx.EVT_MENU(self, 202, self.OnFilePrint)

        menu.Append(203, 'Save Plot...', 'Save current plot')
        wx.EVT_MENU(self, 203, self.OnSaveFile)

        menu.Append(205, 'E&xit', 'Enough of this already!')
        wx.EVT_MENU(self, 205, self.OnFileExit)
        self.mainmenu.Append(menu, '&File')

        menu = wx.Menu()
        menu.Append(210, 'Input &Params', 'Show parameter window')
        wx.EVT_MENU(self, 210, self.OnParameter)
        menu.Append(214, 'Enable &Zoom', 'Enable Mouse Zoom', kind=wx.ITEM_CHECK)
        wx.EVT_MENU(self, 214, self.OnEnableZoom)
        menu.Append(215, 'Enable &Grid', 'Turn on Grid', kind=wx.ITEM_CHECK)
        wx.EVT_MENU(self, 215, self.OnEnableGrid)

        self.mainmenu.Append(menu, '&Plot')

        menu = wx.Menu()
        menu.Append(300, '&Help Index', 'Help of this applicantion...')
        wx.EVT_MENU(self, 300, self.OnHelp)
        self.mainmenu.Append(menu, '&Help')

        self.SetMenuBar(self.mainmenu)

        # A status bar to tell people what's happening
        self.CreateStatusBar(1)

        self.client = PlotCanvas(self)
        # Create mouse event for showing cursor coords in status bar
        wx.EVT_LEFT_DOWN(self.client, self.OnMouseLeftDown)
        self.Show(True)

        self.params=SismoDialog(self,-1,'',style=wx.DEFAULT_FRAME_STYLE)

    def OnMouseLeftDown(self,event):
        s= "Left Mouse Down at Point: (%.4f, %.4f)" % self.client.GetXY(event)
        self.SetStatusText(s)
        event.Skip()

    def OnFilePageSetup(self, event):
        self.client.PageSetup()

    def OnFilePrintPreview(self, event):
        self.client.PrintPreview()

    def OnFilePrint(self, event):
        self.client.Printout()

    def OnSaveFile(self, event):
        self.client.SaveFile()

    def OnFileExit(self, event):
        self.Close()

    def OnFileOpenFile(self, event):
        dlg = wx.FileDialog(self, "Open File...", path, style=wx.OPEN, wildcard = "*")
        if dlg.ShowModal() == wx.ID_OK:
            fname = dlg.GetPath()
            f=open(fname,"r")
            tx1=f.read()
            f.close()
            tx2=eval(tx1)
            self.params.SetValues(tx2)
            self.params.Show(True)
        dlg.Destroy()

    def OnParameter(self, event):
        self.params.Show(True)

    def OnEnableZoom(self, event):
        self.client.SetEnableZoom(event.IsChecked())

    def OnEnableGrid(self, event):
        self.client.SetEnableGrid(event.IsChecked())

    def OnHelp(self, event):
        # Add the Zip filesystem
        wx.FileSystem.AddHandler(wx.ZipFSHandler())
        # Create the viewer
        helpctrl = wx.html.HtmlHelpController()
        # and add the books
        helpctrl.AddBook(path+'SismoViHelp.htb', 1)
        # start it up!
        helpctrl.DisplayContents()
        
def main():
    app=wx.PySimpleApp()
    frame=PlotFrame(None,-1,'SismoVi')
    frame.Show(True)
    app.MainLoop()

path=os.path.dirname(sys.argv[0])
if path!="": path += "/"

if __name__=='__main__':
    main()
